(function(CloudCardViewController, CardViewController, strings) {
    'use strict';

    /**
     * Create Attribution Element
     *
     * @param {string} authorName
     * @param {string} authorPageUrl
     * @param {string} platformLogoSrcUrl
     *
     * @returns {HtmlElement}
     */
    function createAttributionEl(authorName, authorPageUrl, platformLogoSrcUrl, isTenorGif) {
        var containerEl = document.createElement('div'),
            authorLabelEl = document.createElement('span'),
            platformLogoEl = document.createElement('img'),
            anchorStyles,
            attributeTxt;

        anchorStyles = 'overflow: hidden;' +
            'white-space: nowrap;' +
            'color: #999;' +
            'text-decoration: none !important;' +
            'line-height: 30px;' +
            'font-size: 12px;' +
            'cursor: pointer !important;';


        platformLogoEl.setAttribute('src', platformLogoSrcUrl);

        // Use localized string template to create the attribution text nodes
        if (isTenorGif) {
            attributeTxt = strings.mailsdk_tenor_gif_attribution_label;
            attributeTxt = attributeTxt.split('%1$s');
        } else {
            attributeTxt = strings.mailsdk_gif_attribution_label.replace('%1$s', authorName);
            attributeTxt = attributeTxt.split('%2$s');
        }

        if (attributeTxt[0].length > 0) {
            authorLabelEl.appendChild(document.createTextNode(attributeTxt[0]));
        }

        authorLabelEl.appendChild(platformLogoEl);

        if (attributeTxt[1].length > 0) {
            authorLabelEl.appendChild(document.createTextNode(attributeTxt[1]));
        }

        containerEl.innerHTML =
            '<a style="' + anchorStyles + '" href="' + authorPageUrl + '" >' +
                authorLabelEl.outerHTML +
            '</a>';

        return containerEl;
    }

    var AttachmentCardFactory = {
        /**
         * Create a new Cloud Card.
         *
         * @params {string} url
         * @params {string} thumbnailURL
         * @params {string} title
         * @params {string} subtitle
         *
         * @return {CardViewController}
         */
        createCloudCard: function(url, thumbnailURL, title, subtitle) {
            var cloudCardViewController = new CloudCardViewController();

            cloudCardViewController.render({
                url: url,
                thumbnailURL: thumbnailURL,
                title: title,
                subtitle: subtitle
            });

            return cloudCardViewController;
        },

        /**
         * Create a new Gif Card
         *
         * @param {string} gifUrl - source url for the gif image
         * @param {string} authorName - author's name used for attribution
         * @param {string} authorPageUrl - author's webpage used for attribution
         * @param {string} platformLogoSrcUrl - source url for the platform (example: Tumblr), used for attribution
         * @param {boolean} appendAttribution - true if we want to appendAttribution
         *
         * @return {CardViewController}
         */
        createGifCard: function(gifUrl, authorName, authorPageUrl, platformLogoSrcUrl, appendAttribution, isTenorGif) {
            var cardViewController = new CardViewController(),
                gifCardInnerContainerEl = document.createElement('div'),
                gifImgContainerEl = document.createElement('div'),
                removeBtn = cardViewController.createRemoveBtnEl(),
                imageViewController = new ImageViewController();

            gifImgContainerEl.setAttribute('style', 'display: inline-block; position: relative;');
            gifImgContainerEl.innerHTML =
                '<a src="">' +
                    '<img src="' + gifUrl + '" ></img>' +
                '</a>';
            gifImgContainerEl.appendChild(removeBtn);
            // Add image controller for the new added gif
            imageViewController.attach(gifImgContainerEl.getElementsByTagName('img')[0]);
            gifCardInnerContainerEl.appendChild(gifImgContainerEl);

            if (appendAttribution) {
                gifCardInnerContainerEl.appendChild(createAttributionEl(authorName, authorPageUrl, platformLogoSrcUrl, isTenorGif));
            }

            cardViewController.render({
                url: authorPageUrl,
                classNames: [GIF_CARD_EL_CLASS_NAME, IMG_PENDING_CLASS],
                style: 'border: none;' +
                        'width: auto;' +
                        'padding: 0;'
            }, gifCardInnerContainerEl, removeBtn);

            return cardViewController;
        }
    };

    window.AttachmentCardFactory = AttachmentCardFactory;
})(window.CloudCardViewController, window.CardViewController, window.strings);
